#include <stdlib.h>
#include <curses.h>
#include <errno.h>
#include <unistd.h>
#include <sys/time.h>

#define REFRESH_INTERVAL 100000
#define UP 0
#define RIGHT 1
#define DOWN 2
#define LEFT 3

#define STAGE_WIDTH 20
#define STAGE_HEIGHT 20

#define LEFTKEY 'j'
#define RIGHTKEY 'l'

#define FOOD "~"
#define SNAKE1 "O"
#define SNAKE2 "@"

typedef struct
{
  // screen control
  SCREEN* scr;
  WINDOW* win;

  // game status
  int score;
  int lines;
  int cols;
  int* stage;
  int refresh_sec;
  int refresh_usec;
  
  // food info
  int food_num;
  int food_time;
  
  // snake info
  int length;
  int x;
  int y;
  int direction;
}snake_state;

void snake_start(snake_state* state)
{
  struct timeval tv;
  
  // make new terminal
  state->win = initscr();
  nodelay(state->win, TRUE);
  noecho();
  
  // change input mode
  crmode();

  // input stage size
  state->lines = STAGE_WIDTH;
  state->cols = STAGE_HEIGHT;

  // initialize snake state
  state->x = state->cols / 2;
  state->y = state->lines / 2;
  state->length = 5;
  state->direction = UP; 

  // allocate memory for stage
  state->stage = (int *)malloc(sizeof(int)*state->lines*state->cols);
  state->stage = (int *)memset(state->stage, 0, sizeof(int)*state->lines*state->cols);

  gettimeofday(&tv, NULL);
  state->refresh_sec = tv.tv_sec;
  state->refresh_usec = tv.tv_usec;
}

void snake_end(snake_state* state)
{
  // free resources
  if(state)
    {
      endwin();
      if(state->scr)  delscreen(state->scr);
      if(state->stage)  free(state->stage);
      
      free(state);
    }
  // exit program
  exit(0);
}

void snake_stage(snake_state* state)
{
  state->win = subwin(stdscr, state->lines, state->cols, 3, 0);
  if(!state->win)
    {
      snake_end(state);
    }
}

void snake_eat(snake_state* state)
{
  state->score += 100;
  state->food_num--;
  state->length++;
}

int* snake_map_value(snake_state* state, int x, int y)
{
  return (state->stage + state->cols * y + x);
}

void snake_move(snake_state* state)
{
  int* value;
  int x, y;
  
  switch(state->direction)
    {
      case UP: 
        if(*snake_map_value(state, state->x, state->y - 1) == 10000) snake_eat(state);
        else if(*snake_map_value(state, state->x, state->y - 1) > 0) snake_end(state);
        state->y -= 1; 
        break;
      case RIGHT: 
        if(*snake_map_value(state, state->x + 1, state->y) == 10000) snake_eat(state);
        else if(*snake_map_value(state, state->x + 1, state->y) > 0) snake_end(state);
        state->x += 1; 
        break;
      case DOWN: 
        if(*snake_map_value(state, state->x, state->y + 1) == 10000) snake_eat(state);
        else if(*snake_map_value(state, state->x, state->y + 1) > 0) snake_end(state);
        state->y += 1; 
        break;
      case LEFT: 
        if(*snake_map_value(state, state->x - 1, state->y) == 10000) snake_eat(state);
        else if(*snake_map_value(state, state->x - 1, state->y) > 0) snake_end(state);
        state->x -= 1; 
        break;
    }

  if(state->x < 0 || state->x > state->cols)
    {
      snake_end(state);
    }
  if(state->y < 0 || state->y > state->lines)
    {
      snake_end(state);
    }

  for(y = 0; y < state->lines; ++y)
    for(x = 0; x < state->cols; ++x)
      {
        value = state->stage + state->cols * y + x;
        if(*value > 0 && *value < 10000)
          {
            *value = *value - 1;
          }
      }

  value = snake_map_value(state, state->x, state->y);
  *value = state->length;
}

void snake_food(snake_state* state)
{
  int x, y;
  bool isset = true;
  while(isset && state->food_num < 10)
    {
      x = rand() % state->cols;
      y = rand() % state->lines;
      if(*snake_map_value(state, x, y) == 0)
        {
          *snake_map_value(state, x, y) = 10000;
          isset = false;
          state->food_num++;
        }
    }
}

void snake_draw(snake_state* state)
{
  char str[4];
  int x, y, value;
  
  box(state->win, ACS_VLINE, ACS_HLINE);
  printw("SCORE : %d", state->score);

  for(y = 0; y < state->lines; ++y)
    {
      for(x = 0; x < state->cols; ++x)
        {
          value = *(state->stage + state->cols * y + x);
          if(value > 0 && value < 10000)
            {
              if(value % 2) sprintf(str, SNAKE1);
              else sprintf(str, SNAKE2);
              mvwprintw(state->win, y, x, str);
            }
          else if(value == 10000)
            {
              mvwprintw(state->win, y, x, FOOD);
            }
        }
    }
}

void snake_input(snake_state* state)
{
  switch(getch())
    {
      case LEFTKEY:
        state->direction = ((state->direction - 1) + (!state->direction * 4)) % 4;
        break;
      case RIGHTKEY:
        state->direction = (state->direction + 1) % 4;
        break;
    }
}

int snake_loop(snake_state* state)
{
  struct timeval tv;
  snake_stage(state);
  
  while(state->win)
    {
      gettimeofday(&tv, NULL);
      if(tv.tv_sec > state->refresh_sec || tv.tv_usec - state->refresh_usec > REFRESH_INTERVAL)
        {
          clear();
          snake_move(state);
          snake_food(state);
          snake_draw(state);
          state->refresh_sec = tv.tv_sec;
          state->refresh_usec = tv.tv_usec;
        }
      
      refresh();
      snake_input(state);
    }
}

int main()
{
  snake_state* state = NULL;
  state = (snake_state*)malloc(sizeof(snake_state));
  state = (snake_state*)memset(state, 0, sizeof(snake_state));
  
  snake_start(state);
  snake_loop(state);
  snake_end(state);
  return 0;
}

